package atomicstryker.minions.common.jobmanager;

import java.util.ArrayList;

import atomicstryker.minions.common.MinionsCore;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.ChunkPos;
import net.minecraft.world.World;

/**
 * Minion Mod Runnable Tree Scanner class. Finds Trees in a certain area around
 * a first given Tree, returns a List once done.
 * 
 * 
 * @author AtomicStryker
 */

public class TreeScanner implements Runnable
{
    private Minion_Job_TreeHarvest boss;
    private World world;
    private int foundTreeCount;
    private int currentX;
    private int currentZ;
    private int currentMaxX;
    private int currentMaxZ;
    private Block treeBlockID;
    private final ArrayList<ChunkPos> skippableCoords;

    public TreeScanner(Minion_Job_TreeHarvest creator)
    {
        boss = creator;
        skippableCoords = new ArrayList<ChunkPos>();
        foundTreeCount = 0;
        currentMaxX = 0;
        currentMaxZ = 0;
    }

    public void setup(BlockPos coords, World worldObj)
    {
        currentX = coords.getX();
        currentZ = coords.getZ();
        this.world = worldObj;
    }

    @Override
    public void run()
    {
        // System.out.println("AS_TreeScanner starting to run at
        // ["+currentX+"|"+currentZ+"]");
        boolean otherDirection = false;
        while (foundTreeCount < 16 && currentMaxX < 64)
        {
            // iterate length X
            int nextXStop = currentX + (otherDirection ? currentMaxX * -1 : currentMaxX);
            while (currentX != nextXStop)
            {
                checkForTreeAtCoords();
                if (otherDirection)
                {
                    currentX--;
                }
                else
                {
                    currentX++;
                }
            }
            // iterate length Z
            int nextZStop = currentZ + (otherDirection ? currentMaxZ * -1 : currentMaxZ);
            while (currentZ != nextZStop)
            {
                checkForTreeAtCoords();
                if (otherDirection)
                {
                    currentZ--;
                }
                else
                {
                    currentZ++;
                }
            }

            // change movement direction as per search algorithm
            otherDirection = !otherDirection;

            // expand lengths for next run
            currentMaxX++;
            currentMaxZ++;
        }

        MinionsCore.debugPrint("AS_TreeScanner finished work, found: " + foundTreeCount + "; checked length: " + currentMaxX);
        boss.onDoneFindingTrees();
    }

    private void checkForTreeAtCoords()
    {
        if (skippableCoords.contains(new ChunkPos(currentX, currentZ))) // to
                                                                        // fix
                                                                        // more-than-1-block-thick
                                                                        // trees
        {
            return;
        }

        // System.out.println("checkForTreeAtCoords
        // ["+currentX+"|"+currentZ+"]");
        BlockPos y = this.world.getTopSolidOrLiquidBlock(new BlockPos(currentX, 0, currentZ));
        if (y.getY() != -1)
        {
            IBlockState is = this.world.getBlockState(new BlockPos(currentX, y.getY() - 1, currentZ));
            Block ID = is.getBlock();
            if (MinionsCore.instance.foundTreeBlocks.contains(ID))
            {
                IBlockState newState;
                Block newID;
                for (y = y.down();; y = y.down())
                {
                    newState = world.getBlockState(y);
                    newID = newState.getBlock();
                    if (newID != ID)
                    {
                        break;
                    }
                }

                if (newID == Blocks.AIR || newState.getMaterial() == Material.LEAVES || newID.isLeaves(is, world, y))
                {
                    return;
                }
                else
                {
                    onFoundTreeBase(currentX, y.getY() + 1, currentZ);
                }
            }
        }
        Thread.yield();
    }

    private void onFoundTreeBase(int ix, int iy, int iz)
    {
        for (int jx = -1; jx <= 1; jx++)
        {
            for (int jz = -1; jz <= 1; jz++)
            {
                ChunkPos excludeCoords = new ChunkPos(ix + jx, iz + jz);
                if (!skippableCoords.contains(excludeCoords))
                {
                    skippableCoords.add(excludeCoords);
                }
            }
        }

        ArrayList<BlockPos> treeBlockList = new ArrayList<BlockPos>();
        ArrayList<BlockPos> leaveBlockList = new ArrayList<BlockPos>();
        treeBlockID = this.world.getBlockState(new BlockPos(ix, iy, iz)).getBlock();
        indexTargetTree(ix, iy, iz, treeBlockList, leaveBlockList);

        if (treeBlockList.size() > 3)
        {
            foundTreeCount++;
            boss.onFoundTreeBase(ix, iy, iz, treeBlockList, leaveBlockList);
        }
    }

    private void indexTargetTree(int ix, int iy, int iz, ArrayList<BlockPos> treeBlockList, ArrayList<BlockPos> leaveBlockList)
    {
        indexTreeBlockRecursive(ix, iy, iz, treeBlockList, leaveBlockList);
    }

    private void indexTreeBlockRecursive(int ix, int iy, int iz, ArrayList<BlockPos> treeBlockList, ArrayList<BlockPos> leaveBlockList)
    {
        byte one = 1;
        for (int xIter = -one; xIter <= one; xIter++)
        {
            for (int zIter = -one; zIter <= one; zIter++)
            {
                for (int yIter = 0; yIter <= one; yIter++)
                {
                    if (world.getBlockState(new BlockPos(ix + xIter, iy + yIter, iz + zIter)).getBlock() == treeBlockID)
                    {
                        BlockPos coords = new BlockPos(ix + xIter, iy + yIter, iz + zIter);
                        if (!treeBlockList.contains(coords))
                        {
                            treeBlockList.add(coords);
                            findLeavesRecursive(ix + xIter, iy + yIter, iz + zIter, 0, leaveBlockList);
                            indexTreeBlockRecursive(ix + xIter, iy + yIter, iz + zIter, treeBlockList, leaveBlockList);
                        }
                    }
                }
            }
        }
    }

    private void findLeavesRecursive(int ix, int iy, int iz, int fromStem, ArrayList<BlockPos> leaveBlockList)
    {
        for (int xIter = -1; xIter <= 1; xIter++)
        {
            for (int zIter = -1; zIter <= 1; zIter++)
            {
                for (int yIter = 0; yIter <= 1; yIter++)
                {
                    final BlockPos coords = new BlockPos(ix + xIter, iy + yIter, iz + zIter);
                    final IBlockState is = world.getBlockState(coords);
                    final Block b = is.getBlock();
                    if (is.getMaterial() == Material.LEAVES || b.isLeaves(is, world, coords))
                    {
                        if (fromStem < 4)
                        {
                            if (!leaveBlockList.contains(coords))
                            {
                                leaveBlockList.add(coords);
                                findLeavesRecursive(ix + xIter, iy + yIter, iz + zIter, fromStem + 1, leaveBlockList);
                            }
                        }
                    }
                }
            }
        }
    }
}